//
// C++ Implementation: guimainimplementation
//
// Description: Implementation of functionality (buttons/sliders) from the GUI
//
//
// Author:  <Marta Kersten>, (C) 2005
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "guimainimplementation.h"
#include "momentregistrator.h"

guiMainImplementation::guiMainImplementation()
{
	my_fixedViewAxis = 2;
	my_movingViewAxis = 2;
	my_fixedImageFileName = "";
	my_movingImageFileName = "";
	my_fixedImageLoaded = false;
	my_movingImageLoaded = false;
	
	my_fixedImage = 0;
	my_movingImage = 0;
	my_initializedImage = 0; 
	my_registeredImage = 0; 
	
	my_fixedImageSize.Fill(0);
	my_movingImageSize.Fill(0);

	currentFixedSlice = 0; 
	currentMovingSlice = 0; 
	currentResultSlice = 0; 

 	my_fixedLandmarkSpatialObject = LandmarkSpatialObjectType::New();
  	my_movingLandmarkSpatialObject = LandmarkSpatialObjectType::New();
	
	my_finalAffineTransform = AffineTransformType::New();
	my_finalAffineTransform->SetIdentity(); 
	
	my_finalRigidTransform = RigidTransformType::New();
	my_finalRigidTransform->SetIdentity(); 
		
	my_landmarkRegTransform = LandmarkTransformType::New(); 
	my_landmarkRegTransform->SetIdentity(); 

	my_landmarkAffineTransform = AffineTransformType::New(); 
	my_landmarkAffineTransform->SetIdentity();

	my_momentsRegTransform = MomentsTransformType::New(); 
	my_momentsRegTransform->SetIdentity(); 

	my_momentsAffineTransform = AffineTransformType::New(); 
	my_momentsAffineTransform->SetIdentity(); 

	my_rigidTransform = TransformType::New(); 
	my_rigidTransform->SetIdentity(); 

	guiMI = new guiMIImplementation(); 
	guiICP = new guiICPImplementation();
	guiLandReg = new guiLandRegImplementation();
	momentsReg = new MomentRegistrator();  
	guiRigidReg = new guiRigidRegImplementation();
	guiAffineReg = new guiAffineRegImplementation();

	my_interpolator = InterpolatorType::New();
	my_initializer = TransformInitializerType::New(); 
	my_resampleFilter = FilterType::New(); 
	
	SetInitialization(NONE);
	SetRegistrationAlgorithm(NONE);
}

guiMainImplementation::~guiMainImplementation()
{
}

void guiMainImplementation::Show()
{ 
	fixedImageViewer->show();
	movingImageViewer->show();
	
	fixedImageViewer->update();
	movingImageViewer->update();
	
	fixedImageViewer->deactivate();
	movingImageViewer->deactivate(); 
}


// Choose an image to be loaded
void guiMainImplementation::SelectImage(int isFixedImage)
{
	ChangeStatusDisplay("Loading image...");
	char * filename = fl_file_chooser("Choose an image file...","*.mh?","");
	
	if ( !filename )
	{
		return;
	}

	if (isFixedImage == FIXED)
		this->HandleInputImageChange(filename, FIXED);
	
	if (isFixedImage == MOVING)
		this->HandleInputImageChange(filename, MOVING);
}

// Load the image
guiMainImplementation::ImagePointer guiMainImplementation::LoadImage(char * filename ) 
{
	ImageReaderType::Pointer reader;
	ImageType::Pointer image;
	
	reader = ImageReaderType::New();
	reader->SetFileName(filename);

	try
	{
		reader->Update();      
	}
	catch( itk::ExceptionObject &e )
	{
		std::cout<<e<<std::endl;
	}
	catch(...)
	{
		std::cout<<" Unknow exception caught in LoadImage!!!"<<std::endl;
	}
	
	image = reader->GetOutput();
	return image;
}


void guiMainImplementation::HandleInputImageChange(char *fileName, int isFixedImage)
{
	
	ImageType::Pointer image;
	fl_cursor(Fl_Cursor(FL_CURSOR_WAIT), (Fl_Color) 56, (Fl_Color) 255);
	image = LoadImage(fileName);
/*	if ( isFixedImage == FIXED)
		ConvertVTKtoITK(fileName, FIXED); 
	else
		ConvertVTKtoITK(fileName, MOVING); */

	if ( !image )
	{
		fl_cursor(Fl_Cursor(FL_CURSOR_DEFAULT), (Fl_Color) 56, (Fl_Color) 255);
		char *errorString = "Could not load image.";
		ErrorWindow *error = new ErrorWindow(errorString);
		error->Show();
		ChangeStatusDisplay("Error"); 
		return; 
	}

	if ( isFixedImage == FIXED)
	{
	/*	if (my_fixedImageFileName == fileName )
		{
			fl_cursor(Fl_Cursor(FL_CURSOR_DEFAULT), (Fl_Color) 56, (Fl_Color) 255);
			return;
		}*/
		my_fixedImageFileName = fileName;
		my_fixedImage = image;
		my_fixedImageLoaded = true;
		my_fixedImageSize = image->GetLargestPossibleRegion().GetSize(); 
 		fixedSliceSlider->maximum(my_fixedImageSize[2]-1);
    		resultSliceSlider->maximum(my_fixedImageSize[2]-1);
		currentFixedSlice = my_fixedImageSize[2]-1;
		fixedImageViewer->SetInputImage(my_fixedImage.GetPointer());
		fixedImageViewer->viewDetails(false);
		fixedImageViewer->update();
		fixedImageViewer->redraw();
		fixedImageViewer->activate();
 		resultImageViewer->SetInputImage(my_fixedImage.GetPointer());
		currentResultSlice = currentFixedSlice;
	}
	else
	{
	/*	if ( my_movingImageFileName == fileName )
		{
			fl_cursor(Fl_Cursor(FL_CURSOR_DEFAULT), (Fl_Color) 56, (Fl_Color) 255);
			return;
		}*/
		ChangeStatusDisplay("Loading moving image...");
		my_movingImageFileName = fileName;
		my_movingImage = image;
		my_movingImageLoaded = true;
		my_movingImageSize = image->GetLargestPossibleRegion().GetSize();
		movingSliceSlider->maximum(my_movingImageSize[2]-1);
		currentMovingSlice = my_movingImageSize[2]-1;
		movingImageViewer->SetInputImage(my_movingImage.GetPointer());
		movingImageViewer->viewDetails(false);
		movingImageViewer->update();
		movingImageViewer->redraw();
		movingImageViewer->activate();
		resultImageViewer->SetSecondInputImage(my_movingImage.GetPointer());
	}

	if(my_fixedImageLoaded == true && my_movingImageLoaded == true)
	{
		this->SetView(UNREGISTERED);
		unregisteredView->activate(); 
		displayImageChoice->value(2); 
	}
	else if (my_fixedImageLoaded == false && my_movingImageLoaded == true)
	{
		currentResultSlice = currentMovingSlice;
		this->SetView(MOVING);
		movingImageView->activate();
		displayImageChoice->value(1); 
	}
	else
	{
		this->SetView(FIXED);
		fixedImageView->activate();
		displayImageChoice->value(0); 
	}
	
  	resultImageViewer->imageMode(IMG_VAL);
  	resultImageViewer->deactivate();
	resultImageViewer->update();
  	resultImageViewer->redraw();

	this->SetFixedViewAxis(2); 
	fixedViewAxisZ->setonly(); 
	
	this->SetMovingViewAxis(2); 
	movingViewAxisZ->setonly(); 

	fl_cursor(Fl_Cursor(FL_CURSOR_DEFAULT), (Fl_Color) 56, (Fl_Color) 255);
	ChangeStatusDisplay("Image loaded.");
}



void guiMainImplementation::UpdateMovingSliceView()
{
	if ( movingImageViewer->GetInputImage())
    	{
		currentMovingSlice = (unsigned int) movingSliceSlider->value();
    		movingImageViewer->sliceNum(currentMovingSlice);
    		movingImageViewer->update();
    	}
}


void guiMainImplementation::UpdateMovingSliceView(int sliceNum)
{
	if ( movingImageViewer->GetInputImage())
    	{
    		movingImageViewer->sliceNum(sliceNum);
    		movingImageViewer->update();
    	}
	currentMovingSlice = sliceNum; 
}


void guiMainImplementation::UpdateFixedSliceView()
{
	if ( fixedImageViewer->GetInputImage() )
    	{
		currentFixedSlice = (unsigned int) fixedSliceSlider->value();
    		fixedImageViewer->sliceNum(currentFixedSlice);
    		fixedImageViewer->update();
    	}
}


void guiMainImplementation::UpdateFixedSliceView(int sliceNum)
{
	if ( fixedImageViewer->GetInputImage() )
    	{
    		fixedImageViewer->sliceNum(sliceNum);
    		fixedImageViewer->update();
    	}
	currentFixedSlice = sliceNum;
}


void guiMainImplementation::UpdateResultSliceView()
{
  	if ( resultImageViewer->GetInputImage() )
    	{
		currentResultSlice = (unsigned int) resultSliceSlider->value();
    		resultImageViewer->sliceNum(currentResultSlice);
    		resultImageViewer->update();
    	}
}


void guiMainImplementation::UpdateResultSliceView(int sliceNum)
{
  	if ( resultImageViewer->GetInputImage() )
    	{
    		resultImageViewer->sliceNum(sliceNum);
    		resultImageViewer->update();
    	}
	currentResultSlice = sliceNum;
}

void guiMainImplementation::SetMovingViewAxis(unsigned int axis)
{
	my_movingViewAxis = axis;
	
	if (axis == 0) 
	{
		movingViewAxisY->value(0); 
		movingViewAxisZ->value(0); 
	}
	if (axis == 1) 
	{
		movingViewAxisX->value(0); 
		movingViewAxisZ->value(0); 
	}
	if (axis == 2) 
	{
		movingViewAxisX->value(0); 
		movingViewAxisY->value(0); 
	}

	// change orientation to selected axis
	movingImageViewer->orientation(axis);
	
	unsigned int mid = (my_movingImageSize[axis]-1)/2;
	movingSliceSlider->value(mid);
	movingSliceSlider->maximum(my_movingImageSize[axis]-1);
	movingImageViewer->sliceNum(mid);
	
	movingImageViewer->update();
	movingImageViewer->redraw();
}

void guiMainImplementation::SetFixedViewAxis(unsigned int axis)
{
	my_fixedViewAxis = axis;
	
	if (axis == 0) 
	{
		fixedViewAxisY->value(0); 
		fixedViewAxisZ->value(0); 
	}
	if (axis == 1) 
	{
		fixedViewAxisX->value(0); 
		fixedViewAxisZ->value(0); 
	}
	if (axis == 2) 
	{
		fixedViewAxisX->value(0); 
		fixedViewAxisY->value(0); 
	}
	// change orientation to selected axis
	fixedImageViewer->orientation(axis);
	resultImageViewer->orientation(axis);
	
	// update slider and slice number in viewer to middle slice
	unsigned int mid = (my_fixedImageSize[axis]-1)/2;
	resultSliceSlider->value(mid);
	resultSliceSlider->maximum(my_fixedImageSize[axis]-1);
	fixedSliceSlider->value(mid);
	fixedSliceSlider->maximum(my_fixedImageSize[axis]-1);
	resultImageViewer->sliceNum(mid);
	fixedImageViewer->sliceNum(mid);

	fixedImageViewer->update();
	resultImageViewer->update();
	
	fixedImageViewer->redraw();
	resultImageViewer->redraw();
}

void guiMainImplementation::SetView(unsigned int view)
{
	if(view == FIXED)
	{
		if(my_fixedImageLoaded)
		{
			resultImageViewer->ClearSecondInputImage();
			resultImageViewer->SetInputImage(my_fixedImage.GetPointer());
		}
	}
	else if(view == MOVING)
	{
		if(!my_movingImageLoaded)
			return;
		else
		{
			resultImageViewer->ClearSecondInputImage();
			resultImageViewer->SetInputImage(my_movingImage.GetPointer());
		}
	}
	else if(view == REGISTERED)
	{
		if(!my_registeredImage)
			return;
		else
		{
			resultImageViewer->SetInputImage(my_fixedImage.GetPointer());
			resultImageViewer->SetSecondInputImage(my_registeredImage.GetPointer());
		}
	}
	else if(view == UNREGISTERED)
	{
		if(!my_movingImageLoaded)
			return;
		else
		{
			resultImageViewer->SetInputImage(my_fixedImage.GetPointer());
			resultImageViewer->SetSecondInputImage(my_movingImage.GetPointer());
		}
	}
	
	else if(view == INITIALIZED)
	{
		if(!my_initializedImage)
			return;
		else
		{
			resultImageViewer->SetInputImage(my_fixedImage.GetPointer());
			resultImageViewer->SetSecondInputImage(my_initializedImage.GetPointer());
		}
	}

	resultImageViewer->deactivate(); 	
	UpdateResultSliceView(currentResultSlice); 
	resultImageViewer->imageMode(IMG_VAL);
  	resultImageViewer->update();
  	resultImageViewer->redraw();
  	resultImageViewer->activate();
  
}
void guiMainImplementation::AddLandmark(Fl_Widget *w)
{
	SliceViewer <short> *view;
	Fl_Group* grp = (Fl_Group*) w;
	
	if (grp == fixedImageGroup)
	{
		numFixedLandmarks++;
		view = fixedImageViewer;
		view->AddLandmark(numFixedLandmarks, numFixedLandmarks);
		my_fixedLandmarkSpatialObject->SetPoints(view->GetLandmarkList());
	}
	else
	{	
		numMovingLandmarks++; 
		view = movingImageViewer;
		view->AddLandmark(numMovingLandmarks, numMovingLandmarks); 
    		my_movingLandmarkSpatialObject->SetPoints(view->GetLandmarkList());
	}


	

}


void guiMainImplementation::LoadLandmarks( unsigned int type)
{
	const char * filename = 
	fl_file_chooser("Choose a file for landmarks...","*.lnd","");

	if( filename )
	{
		typedef itk::SpatialObjectReader<> ReaderType;
		
		ReaderType::Pointer reader = ReaderType::New();
		reader->SetFileName(filename);
		reader->Update();
		ReaderType::GroupPointer group = reader->GetGroup();
		ReaderType::GroupType::ChildrenListType* children = group->GetChildren();
		LandmarkSpatialObjectType::Pointer landmarks = 
		dynamic_cast< LandmarkSpatialObjectType* >(
			(*(children->begin())).GetPointer());
	
		if (type == MOVING)
		{
			my_movingLandmarkSpatialObject->SetPoints( landmarks->GetPoints() );
		
			if ( my_movingLandmarkSpatialObject->GetPoints().size() > 0 )
			{
				movingImageViewer->SetLandmarkList( 
					& my_movingLandmarkSpatialObject->GetPoints());
				numMovingLandmarks += my_movingLandmarkSpatialObject->GetPoints().size();
				std::cout<<"num movingpoints: "<<numMovingLandmarks<<std::endl;
			}
		}
		else
		{
			my_fixedLandmarkSpatialObject->SetPoints( landmarks->GetPoints() );
			
			if ( my_fixedLandmarkSpatialObject->GetPoints().size() > 0 )
			{
				fixedImageViewer->SetLandmarkList( 
					& my_fixedLandmarkSpatialObject->GetPoints());
				numFixedLandmarks += my_fixedLandmarkSpatialObject->GetPoints().size();
			std::cout<<"num movingpoints: "<<numMovingLandmarks<<std::endl;
			}
		
		}     
	}
	
}

void guiMainImplementation::SaveLandmarks( unsigned int type )
{
	const char * filename = 
	fl_file_chooser("Choose a file for landmarks...","*.lnd","");
	
	if( !filename )
	{
		return;
	}
	
	typedef itk::LandmarkSpatialObject< 3 > ObjectType;
	typedef itk::SpatialObjectWriter< > WriterType;
	typedef itk::GroupSpatialObject< > GroupType;
	
	WriterType::Pointer writer = WriterType::New();
	writer->SetFileName(filename);
	ObjectType::Pointer landmarks = ObjectType::New();
	
	if (type == MOVING)
	{
		landmarks->SetPoints(movingImageViewer->GetLandmarkList());
	}
	else
	{
		landmarks->SetPoints(fixedImageViewer->GetLandmarkList());
	}

	writer->SetInput(landmarks);
	writer->Update();
}

void guiMainImplementation::ClearLandmarks(unsigned int viewWindow)
{

	SliceViewer<short> *view;

  	if (viewWindow == FIXED)
	{
    		view = fixedImageViewer;
		for ( unsigned int i = 0; i < numFixedLandmarks+1; ++i )
		{
			if ( view->IsLandmarkAvailable(i) )
			{
				view->DeleteLandmark(i);
			}		
		}
		numFixedLandmarks = 0;
	}
	else
	{
    		view = movingImageViewer;
		for ( unsigned int i = 0; i < numMovingLandmarks+1; ++i )
		{
			if ( view->IsLandmarkAvailable(i) )
			{
				view->DeleteLandmark(i);
			}
				
		}
		numMovingLandmarks = 0;
	} 
	
	
	if ( view == movingImageViewer )
	{
		my_movingLandmarkSpatialObject->
		SetPoints( movingImageViewer->GetLandmarkList() );
		movingImageViewer->update();
		movingImageViewer->redraw();
		movingImageViewer->activate();
	}
	if ( view == fixedImageViewer )
	{
		my_fixedLandmarkSpatialObject->
		SetPoints( fixedImageViewer->GetLandmarkList() );
		fixedImageViewer->update();
		fixedImageViewer->redraw();
		fixedImageViewer->activate();
	}
	

}

void guiMainImplementation::SetInitialization(unsigned int init)
{
	if(init == NONE)
	{
		return;
	}
	else if(init == MOMENTS)
	{
		my_initializationAlgorithm = MOMENTS; 
	}
	else if(init == LANDMARK)
	{
		my_initializationAlgorithm = LANDMARK; 
	}
}

void guiMainImplementation::SetRegistrationAlgorithm(unsigned int algo)
{
	if (algo == ICP)
	{
		my_registrationAlgorithm = ICP; 
		guiICP->SetUp(); 
		guiICP->Show();
	}
	else if(algo == RIGID)
	{
		my_registrationAlgorithm = RIGID;
	}
	else if(algo == AFFINE)
	{
		my_registrationAlgorithm = AFFINE;
	}
	
}

void guiMainImplementation::Register()
{
	bool successful = false;
	if(my_registrationAlgorithm == RIGID)
	{
		// perform rigid registration
		ChangeStatusDisplay("Performing rigid registration...");
		
		guiRigidReg->rigidReg->my_fixedImage = my_fixedImage;
		guiRigidReg->rigidReg->my_movingImage = my_movingImage;
	
		// get initialized transform
		RigidParametersType initParams = my_finalRigidTransform->GetParameters(); 
		itk::Point<double, 3> center;
		if(my_initializationAlgorithm == LANDMARK)
		{
			LandmarkParametersType lndParams = my_landmarkRegTransform->GetParameters();
			initParams.Fill(0); 
		
			initParams[0] = lndParams[0];
			initParams[1] = lndParams[1];
			initParams[2] = lndParams[2];
			initParams[3] = lndParams[3];
			initParams[4] = lndParams[4];
			initParams[5] = lndParams[5];
		
			center = my_landmarkRegTransform->GetCenter();	
		}
		else if(my_initializationAlgorithm == MOMENTS)
		{
			itk::Versor<double> v;
			v.Set(my_momentsRegTransform->GetMatrix()); 
			initParams[0] = v.GetX(); 
			initParams[1] = v.GetY(); 
			initParams[2] = v.GetZ(); 
			initParams[3] = my_momentsRegTransform->GetTranslation()[0];
			initParams[4] = my_momentsRegTransform->GetTranslation()[1];
			initParams[5] = my_momentsRegTransform->GetTranslation()[2];
			center = my_momentsRegTransform->GetCenter(); 
		}
		guiRigidReg->rigidReg->my_transform->SetCenter(center);
		guiRigidReg->rigidReg->my_initialTransformParameters = initParams;
			
		successful = guiRigidReg->rigidReg->Register();
		
		my_rigidTransform = guiRigidReg->rigidReg->my_finalTransform;
		my_finalAffineTransform->SetIdentity();
		my_finalAffineTransform->SetCenter(my_rigidTransform->GetCenter());
		my_finalAffineTransform->SetMatrix( my_rigidTransform->GetRotationMatrix());
		my_finalAffineTransform->SetOffset(my_rigidTransform->GetOffset());
		
  		std::cout << "DEBUG: Rigid: FINAL affine transform: "  << std::endl
		<< my_finalRigidTransform->GetParameters() << std::endl
		<< "   Offset = " << my_finalRigidTransform->GetOffset() << std::endl
		<< "   Center = " << my_finalRigidTransform->GetCenter() << std::endl
		<< "   Transform Parameters = " << guiRigidReg->rigidReg->my_registration->GetLastTransformParameters() << std::endl
		<< "   Rotation Matrix = " << my_finalRigidTransform->GetRotationMatrix() << std::endl
		<< "   Translation = " << my_finalRigidTransform->GetTranslation() << std::endl;		

		if (successful)
		{	
			GetRegistrationTransform();
			ResampleFinalRegisteredImage(); 
		}
	}

	else if(my_registrationAlgorithm == AFFINE)
	{
		//perform affine registration
		ChangeStatusDisplay("Performing affine registration...");

		guiAffineReg->affineReg->my_fixedImage = my_fixedImage;
		guiAffineReg->affineReg->my_movingImage = my_movingImage;
		
		AffineParametersType initParams = my_finalAffineTransform->GetParameters(); 	
		itk::Point<double, 3> center;

		if (my_initializationAlgorithm == LANDMARK)
		{
			LandmarkParametersType lndParams = my_landmarkAffineTransform->GetParameters();
			initParams.Fill(0); 
			initParams.SetSize(15);
			initParams[0] = lndParams[0];
			initParams[1] = lndParams[1];
			initParams[2] = lndParams[2];
			initParams[3] = lndParams[3];
			initParams[4] = lndParams[4];
			initParams[5] = lndParams[5];
			initParams[6] = 1;
			initParams[7] = 1;
			initParams[8] = 1;
			initParams[9] = 1;
			initParams[10]= 1;
			initParams[11] = 1;
			initParams[12] = 1;
			initParams[13]= 1;
			initParams[14] = 1;

			center = my_landmarkAffineTransform->GetCenter();	
		}

		guiAffineReg->affineReg->my_transform->SetCenter(center);
		guiAffineReg->affineReg->my_initialTransformParameters = initParams;
		
		successful = guiAffineReg->affineReg->Register();

		if (successful)
		{	
			GetRegistrationTransform();
			ResampleFinalRegisteredImage(); 
		}
	}
	else if (my_registrationAlgorithm == ICP)
	{
		ChangeStatusDisplay("Registering using icp...");
		successful = guiICP->Register(); 
	}
	else if (my_registrationAlgorithm == NONE)
	{
		fl_cursor(Fl_Cursor(FL_CURSOR_DEFAULT), (Fl_Color) 56, (Fl_Color) 255);
		return;
	}
	if (successful)
	{	
		registeredView->activate();
		this->SetView(REGISTERED); 
		displayImageChoice->value(4); 
		ChangeStatusDisplay("Registration Done.");
	}	
	fl_cursor(Fl_Cursor(FL_CURSOR_DEFAULT), (Fl_Color) 56, (Fl_Color) 255);
	
}

/*void guiMainImplementation::GetInitialTransform()
{
	if(my_initializationAlgorithm == LANDMARK)
	{
		my_landmarkRegTransform->
		RigidParametersType initParams;
		initParams = my_finalRigidTransform->GetParameters();
		
		RigidParametersType lndmrkParams;
		lndmrkParams = guiLandReg->landReg->my_landmarkOptimizer->GetCurrentPosition();
		initParams[0] = lndmrkParams[0];
		initParams[1] = lndmrkParams[1];
		initParams[2] = lndmrkParams[2];
		initParams[3] = lndmrkParams[3];	
		initParams[4] = lndmrkParams[4];
		initParams[5] = lndmrkParams[5];	
	
		guiRigidReg->rigidReg->my_registration->SetInitialTransformParameters(initParams); 
		itk::Point<double, 3> center = guiLandReg->landReg->my_transform->GetCenter();
		guiRigidReg->rigidReg->my_transform->SetCenter(center); 
	}
}*/
 
void guiMainImplementation::Initialize() 
{
	fl_cursor(Fl_Cursor(FL_CURSOR_WAIT), (Fl_Color) 56, (Fl_Color) 255);

	bool successful = false;

	if (my_initializationAlgorithm == NONE)
	{
		this->Register();
		fl_cursor(Fl_Cursor(FL_CURSOR_DEFAULT), (Fl_Color) 56, (Fl_Color) 255);
		return;
	}


	if (my_initializationAlgorithm == LANDMARK )
	{
		ChangeStatusDisplay("Initializaing using landmarks...");
		
		if (my_fixedLandmarkSpatialObject->GetPoints().size() == 0)
		{
			fl_cursor(Fl_Cursor(FL_CURSOR_DEFAULT), (Fl_Color) 56, (Fl_Color) 255);
			char *errorString = "There are no landmarks for the fixed image. Please load or select landmarks.";
			ErrorWindow *error = new ErrorWindow(errorString);
			error->Show();
			ChangeStatusDisplay("Error"); 
			return; 
		}

		if (my_movingLandmarkSpatialObject->GetPoints().size() == 0)
		{
			fl_cursor(Fl_Cursor(FL_CURSOR_DEFAULT), (Fl_Color) 56, (Fl_Color) 255);
			char *errorString = "There are no landmarks for the moving image. Please load or select landmarks.";
			ErrorWindow *error = new ErrorWindow(errorString);
			error->Show();
			ChangeStatusDisplay("Error"); 
			return; 
		}
	
		guiLandReg->landReg->my_fixedLandmarkSet->SetPoints(my_fixedLandmarkSpatialObject->GetPoints());
		guiLandReg->landReg->my_movingLandmarkSet->SetPoints(my_movingLandmarkSpatialObject->GetPoints());
		
		guiLandReg->landReg->my_fixedImage = my_fixedImage;
		guiLandReg->landReg->my_movingImage = my_movingImage;	
		
		successful = guiLandReg->landReg->InitializeRegistration(); 

		if (successful)
		{
			my_landmarkRegTransform = guiLandReg->landReg->my_transform;

			std::cout << "DEBUG: Landmark: FINAL Rigid transform: "  << std::endl
			<< my_landmarkRegTransform->GetParameters() << std::endl;
		
			my_landmarkAffineTransform->SetIdentity();
			my_landmarkAffineTransform->SetCenter(guiLandReg->landReg->my_transform->GetCenter()); 
			my_landmarkAffineTransform->SetMatrix(guiLandReg->landReg->my_transform->GetRotationMatrix()); 
			my_landmarkAffineTransform->SetOffset(guiLandReg->landReg->my_transform->GetOffset()); 

  			std::cout << "DEBUG: Landmark: FINAL affine transform: "  << std::endl
            		<< my_landmarkAffineTransform->GetParameters() << std::endl
           		<< "   Offset = " << my_landmarkAffineTransform->GetOffset() << std::endl
           		<< "   Center = " << my_landmarkAffineTransform->GetCenter() << std::endl;

			my_resampleFilter = FilterType::New(); 
			my_resampleFilter->SetInput(my_movingImage); 
			my_resampleFilter->SetOutputOrigin(my_fixedImage->GetOrigin());my_resampleFilter->SetOutputSpacing(my_fixedImage->GetSpacing()); 
			my_resampleFilter->SetSize(my_fixedImage->GetLargestPossibleRegion().GetSize());
   		
			AffineTransformType::Pointer inverse = AffineTransformType::New();
			my_landmarkAffineTransform->GetInverse(inverse); 
  			my_resampleFilter->SetTransform(inverse);
  			my_resampleFilter->Update();
			my_initializedImage = my_resampleFilter->GetOutput();
		}
		ChangeStatusDisplay("Initialization done.");
	}
	/*else if (my_RegistrationAlgorithm == MUTUALINFO)
	{
		ChangeStatusDisplay("Initializaing using mutual information...");
		guiMI->MIReg->my_fixedImage = my_fixedImage; 
		guiMI->MIReg->my_movingImage = my_movingImage;  

		successful = guiMI->MIReg->Register(); 
			
		if(successful)
		{	
			GetRegTransform();
			ResampleFinalRegisteredImage(); 
		}
		ChangeStatusDisplay("Initialization Done.");
	}*/
	else if (my_initializationAlgorithm == MOMENTS)
	{
		ChangeStatusDisplay("Initializing using moments...");
		momentsReg->my_fixedImage = my_fixedImage; 
		momentsReg->my_movingImage = my_movingImage;  

		successful = momentsReg->initializeRegistration();
		my_momentsRegTransform = momentsReg->my_transform;
		my_momentsAffineTransform->SetIdentity(); 
		my_momentsAffineTransform->SetCenter(my_momentsRegTransform->GetCenter()); 
		my_momentsAffineTransform->SetMatrix(my_momentsRegTransform->GetMatrix()); 
		my_momentsAffineTransform->SetOffset(my_momentsRegTransform->GetOffset()); 
		my_finalAffineTransform = my_momentsAffineTransform; 

		InterpolatorType::Pointer interpolator = InterpolatorType::New();
		interpolator->SetInputImage(my_movingImage);
		
		my_resampleFilter = FilterType::New(); 
		my_resampleFilter->SetInput(my_movingImage); 
		my_resampleFilter->SetInterpolator(interpolator.GetPointer()); 
		my_resampleFilter->SetOutputOrigin(my_fixedImage->GetOrigin()); 
		my_resampleFilter->SetOutputSpacing(my_fixedImage->GetSpacing()); 
		my_resampleFilter->SetSize(my_fixedImage->GetLargestPossibleRegion().GetSize());
	
		AffineTransformType::Pointer inverse = AffineTransformType::New();
		my_momentsAffineTransform->GetInverse(inverse); 
		my_resampleFilter->SetTransform(inverse);
		my_resampleFilter->Update();
	
		my_initializedImage = my_resampleFilter->GetOutput();
	
		ChangeStatusDisplay("Initialization Done.");
	}

	if(successful)
	{
		initializedView->activate();
		this->SetView(INITIALIZED); 
		displayImageChoice->value(3); 
		std::cout<<"my registration algorithm: "<< my_registrationAlgorithm<<std::endl;
		if(my_registrationAlgorithm != NONE)
			Register();
	}
	fl_cursor(Fl_Cursor(FL_CURSOR_DEFAULT), (Fl_Color) 56, (Fl_Color) 255);


}

void guiMainImplementation::GetRegistrationTransform()
{
	double xTransVal = 0; 
	double yTransVal = 0; 
	double zTransVal = 0;
	double xRotVal = 0; 
	double yRotVal = 0; 
	double zRotVal = 0;
	
	if(my_registrationAlgorithm == RIGID)
	{	
		xTransVal = this->guiRigidReg->rigidReg->my_registration->GetLastTransformParameters()[0];
		yTransVal = this->guiRigidReg->rigidReg->my_registration->GetLastTransformParameters()[1];
		zTransVal = this->guiRigidReg->rigidReg->my_registration->GetLastTransformParameters()[2];
			
		xRotVal = this->guiRigidReg->rigidReg->my_registration->GetLastTransformParameters()[3];
		yRotVal = this->guiRigidReg->rigidReg->my_registration->GetLastTransformParameters()[4];
		zRotVal = this->guiRigidReg->rigidReg->my_registration->GetLastTransformParameters()[5];
	}
	if(my_registrationAlgorithm == AFFINE)
	{	
		xTransVal = this->guiAffineReg->affineReg->my_registration->GetLastTransformParameters()[0];
		yTransVal = this->guiAffineReg->affineReg->my_registration->GetLastTransformParameters()[1];
		zTransVal = this->guiAffineReg->affineReg->my_registration->GetLastTransformParameters()[2];
			
		xRotVal = this->guiAffineReg->affineReg->my_registration->GetLastTransformParameters()[3];
		yRotVal = this->guiAffineReg->affineReg->my_registration->GetLastTransformParameters()[4];
		zRotVal = this->guiAffineReg->affineReg->my_registration->GetLastTransformParameters()[5];
	}
	else if(my_registrationAlgorithm == ICP)
	{
		xTransVal = this->guiICP->icp->my_registration->GetLastTransformParameters()[0];
		yTransVal = this->guiICP->icp->my_registration->GetLastTransformParameters()[1];
		zTransVal = this->guiICP->icp->my_registration->GetLastTransformParameters()[2];
			
		xRotVal = this->guiICP->icp->my_registration->GetLastTransformParameters()[3];
		yRotVal = this->guiICP->icp->my_registration->GetLastTransformParameters()[4];
		zRotVal = this->guiICP->icp->my_registration->GetLastTransformParameters()[5];
	}
	
	char xTransStr[15];
	char yTransStr[15];
	char zTransStr[15];

	sprintf(xTransStr, "%5.2f", xTransVal);
	sprintf(yTransStr, "%5.2f", yTransVal);
	sprintf(zTransStr, "%5.2f", zTransVal);
	
	std::cout<< "x : " << xTransStr << " y: " << yTransStr << " z: " << zTransStr <<std::endl;

	char xRotStr[15];
	char yRotStr[15];
	char zRotStr[15];
	
	sprintf(xRotStr, "%5.2f", xRotVal);
	sprintf(yRotStr, "%5.2f", yRotVal);	
	sprintf(zRotStr, "%5.2f", zRotVal);
	
	xTrans->value(xTransStr);
	yTrans->value(yTransStr);
	zTrans->value(zTransStr);
	
	xRot->value(xRotStr);
	yRot->value(yRotStr);
	zRot->value(zRotStr);

	std::cout<< "x : " << xRotStr << " y: " << yRotStr << " z: " << zRotStr <<std::endl;
}

void guiMainImplementation::GetInitializationTransform()
{
	double xTransVal = 0; 
	double yTransVal = 0; 
	double zTransVal = 0;
	double xRotVal = 0; 
	double yRotVal = 0; 
	double zRotVal = 0;
	
	if(my_initializationAlgorithm == LANDMARK)
	{
		xTransVal = this->guiLandReg->landReg->my_landmarkOptimizer->GetCurrentPosition()[0];
		yTransVal = this->guiLandReg->landReg->my_registration->GetLastTransformParameters()[1];
		zTransVal = this->guiLandReg->landReg->my_registration->GetLastTransformParameters()[2];
			
		xRotVal = this->guiLandReg->landReg->my_registration->GetLastTransformParameters()[3];
		yRotVal = this->guiLandReg->landReg->my_registration->GetLastTransformParameters()[4];
		zRotVal = this->guiLandReg->landReg->my_registration->GetLastTransformParameters()[5];
	}
/*	else if(my_RegistrationAlgorithm == MUTUALINFO)
	{
		xTransVal = this->guiMI->MIReg->my_registrator->GetLastTransformParameters()[0];
		yTransVal = this->guiMI->MIReg->my_registrator->GetLastTransformParameters()[1];
		zTransVal = this->guiMI->MIReg->my_registrator->GetLastTransformParameters()[2];
			
		xRotVal = this->guiMI->MIReg->my_registrator->GetLastTransformParameters()[3];
		yRotVal = this->guiMI->MIReg->my_registrator->GetLastTransformParameters()[4];
		zRotVal = this->guiMI->MIReg->my_registrator->GetLastTransformParameters()[5];
	}*/
	
	else if(my_initializationAlgorithm == MOMENTS)
	{
		xTransVal = this->momentsReg->my_registration->GetLastTransformParameters()[0];
		yTransVal = this->momentsReg->my_registration->GetLastTransformParameters()[1];
		zTransVal = this->momentsReg->my_registration->GetLastTransformParameters()[2];
			
		xRotVal = this->momentsReg->my_registration->GetLastTransformParameters()[3];
		yRotVal = this->momentsReg->my_registration->GetLastTransformParameters()[4];
		zRotVal = this->momentsReg->my_registration->GetLastTransformParameters()[5];
	}
	char xTransStr[15];
	char yTransStr[15];
	char zTransStr[15];

	sprintf(xTransStr, "%5.2f", xTransVal);
	sprintf(yTransStr, "%5.2f", yTransVal);
	sprintf(zTransStr, "%5.2f", zTransVal);
	
	std::cout<< "x : " << xTransStr << " y: " << yTransStr << " z: " << zTransStr <<std::endl;

	char xRotStr[15];
	char yRotStr[15];
	char zRotStr[15];
	
	sprintf(xRotStr, "%5.2f", xRotVal);
	sprintf(yRotStr, "%5.2f", yRotVal);	
	sprintf(zRotStr, "%5.2f", zRotVal);
	
	xTrans->value(xTransStr);
	yTrans->value(yTransStr);
	zTrans->value(zTransStr);
	
	xRot->value(xRotStr);
	yRot->value(yRotStr);
	zRot->value(zRotStr);

	std::cout<< "x : " << xRotStr << " y: " << yRotStr << " z: " << zRotStr <<std::endl;
}

void guiMainImplementation::ResampleFinalRegisteredImage( )
{

	std::cout << "In resample filter... "<<std::endl;
  	my_interpolator->SetInputImage(my_movingImage);

	my_resampleFilter = FilterType::New(); 
	my_resampleFilter->SetOutputOrigin(my_fixedImage->GetOrigin()); 
	my_resampleFilter->SetOutputSpacing(my_fixedImage->GetSpacing()); 
	my_resampleFilter->SetDefaultPixelValue(100); 
	my_resampleFilter->SetInterpolator(my_interpolator.GetPointer()); 
	my_resampleFilter->SetInput(my_movingImage); 
	my_resampleFilter->SetSize(my_fixedImage->GetLargestPossibleRegion().GetSize());	
	
	if (my_registrationAlgorithm == RIGID )
	{
  		// my_finalRigidTransform = RigidTransformType::New();	
  		RigidTransformType::Pointer inverse = RigidTransformType::New();
		guiRigidReg->rigidReg->my_finalTransform->GetInverse(inverse); 
  		my_resampleFilter->SetTransform(inverse);
  		my_resampleFilter->Update();
	}
	if (my_registrationAlgorithm == AFFINE )
	{ 
		// my_finalAffineTransform = AffineTransformType::New();	
  		AffineTransformType::Pointer inverse = AffineTransformType::New();
  		guiAffineReg->affineReg->my_finalTransform->GetInverse(inverse); 
  		my_resampleFilter->SetTransform(inverse);
  		my_resampleFilter->Update();
	}

	
	/*if (my_initializationAlgorithm == LANDMARK )
	{
  		guiLandReg->landReg->my_transform->GetInverse(inverse); 
  		my_resampleFilter->SetTransform(inverse);
  		my_resampleFilter->Update();
	}
	else if (my_initializationAlgorithm == MUTUALINFO)
	{
		guiMI->MIReg->my_transform->GetInverse(inverse); 	
  		my_resampleFilter->SetTransform(inverse);
  		my_resampleFilter->Update();
	}

	else if (my_initializationAlgorithm == MOMENTS); 
	{			
		this->momentsReg->my_finalTransform->GetInverse(inverse);
  		my_resampleFilter->SetTransform(inverse);
		my_resampleFilter->Update();

	}*/
		std::cout << "Resample Transform: " << my_resampleFilter->GetTransform()->GetParameters() << std::endl;

		my_registeredImage = my_resampleFilter->GetOutput();
		resultImageViewer->deactivate();
		resultImageViewer->SetSecondInputImage(my_registeredImage.GetPointer());
		resultImageViewer->update();
		resultImageViewer->redraw();
		resultImageViewer->activate();

		my_extractFilter = ExtractImageFilterType::New(); 
		my_extractFilter->SetInput(my_fixedImage.GetPointer()); 

		ImageType::RegionType inputRegion = my_fixedImage->GetLargestPossibleRegion();		
		ImageType::SizeType size = inputRegion.GetSize(); 
		size[2] = 0;

		ImageType::RegionType desiredRegion;
		desiredRegion.SetSize(size); 

		ImageType::IndexType start = inputRegion.GetIndex();	
		start[2] = 163;
		desiredRegion.SetIndex(start);
		
		my_extractFilter->SetExtractionRegion(desiredRegion);
		my_extractFilter->Update(); 

		Image2DWriterType::Pointer my_writer;
		my_writer = Image2DWriterType::New();
		my_writer->SetFileName("output1.dcm");
		my_writer->SetInput(my_extractFilter->GetOutput());  
		my_writer->Update(); 

		my_checkerboardFilter = CheckerBoardFilterType::New();
		my_checkerboardFilter->SetInput1(my_extractFilter->GetOutput()); 

		my_extractFilter = ExtractImageFilterType::New(); 
		my_extractFilter->SetInput(my_registeredImage.GetPointer()); 
		start = inputRegion.GetIndex();	
		start[2] = 8;
		desiredRegion.SetIndex(start);
	
		my_extractFilter->SetExtractionRegion(desiredRegion);
		my_extractFilter->Update(); 
 
		my_writer->SetInput(my_extractFilter->GetOutput()); 
		my_writer->SetFileName("output2.dcm");
		my_writer->Update();

		my_checkerboardFilter->SetInput2(my_extractFilter->GetOutput()); 
		my_checkerboardFilter->Update(); 

		my_writer->SetInput(my_checkerboardFilter->GetOutput()); 
		my_writer->SetFileName("checkboard.dcm");
		my_writer->Update();
	
}


void guiMainImplementation::ChangeStatusDisplay(const char* message)
{
	statusDisplay->value(message);
	std::cout << message << std::endl;
	Fl::flush();
	Fl::redraw();
}


/*void guiMainImplementation::ConvertVTKtoITK(char* filename, int fixed)
{
	// read SLC file
 	vtkSLCReader *slcReader = vtkSLCReader::New();
	slcReader->SetFileName(filename); 
	slcReader->Update(); 
	
	vtkImageData *volume = slcReader->GetOutput();
	volume->PrintSelf(std::cout, 4);

	typedef  itk::Image<unsigned char, 3> ImageInputType;
	typedef  itk::Image<short,3>  ImageOutputType;
	typedef  itk::VTKImageToImageFilter<ImageInputType>    VtkToItkConnectorType;
 	
	VtkToItkConnectorType::Pointer filter = VtkToItkConnectorType::New();
	filter->SetInput(volume);
	filter->Update();

	ImageInputType::ConstPointer itkImage = filter->GetOutput();
	
	typedef itk::CastImageFilter <ImageInputType, ImageOutputType> CastFilterType; 
	CastFilterType::Pointer castFilter;
     
	typedef itk::ImageFileWriter<ImageInputType> WriterType; 
	WriterType::Pointer writer = WriterType::New();
	
	
//	writer->SetFileName("output");
//	writer->SetInput(filter->GetOutput());
//	writer->Update(); 

	castFilter->SetInput(filter->GetOutput()); 
	castFilter->Update(); 

	if(fixed == FIXED)
	{
		std::cout<<"set fixed image"<<std::endl;
		my_fixedImage = castFilter->GetOutput();
	}
	else
		my_movingImage = castFilter->GetOutput();

 
	// set up VTK export
	vtkImageExport* exporter = vtkImageExport::New();
	exporter->SetInput(volume);

	// set up ITK import
 	typedef itk::VTKImageImport<ImageType> ImageImportType;
  	ImageImportType *importer = ImageImportType::New();

	importer->SetUpdateInformationCallback(exporter->GetUpdateInformationCallback());
	importer->SetPipelineModifiedCallback(exporter->GetPipelineModifiedCallback());
	importer->SetWholeExtentCallback(exporter->GetWholeExtentCallback());
	importer->SetSpacingCallback(exporter->GetSpacingCallback());
	importer->SetOriginCallback(exporter->GetOriginCallback());
	importer->SetScalarTypeCallback(exporter->GetScalarTypeCallback());
	importer->SetNumberOfComponentsCallback(exporter->GetNumberOfComponentsCallback());
	importer->SetPropagateUpdateExtentCallback(exporter->GetPropagateUpdateExtentCallback());
	importer->SetUpdateDataCallback(exporter->GetUpdateDataCallback());
	importer->SetDataExtentCallback(exporter->GetDataExtentCallback());
	importer->SetBufferPointerCallback(exporter->GetBufferPointerCallback());
	importer->SetCallbackUserData(exporter->GetCallbackUserData());

	// import image
	if(fixed)
		my_fixedImage = importer->GetOutput();
	else
		my_movingImage = importer->GetOutput(); 
}*/


void guiMainImplementation::ShowAdvancedInitOptions()
{
	if (my_initializationAlgorithm == LANDMARK)
	{
		guiLandReg->SetUp();
		guiLandReg->Show();
	}

}
void guiMainImplementation::ShowAdvancedRegOptions()
{
	if(my_registrationAlgorithm == ICP)
	{
		guiICP->SetUp();
		guiICP->Show(); 
	}
	else if(my_registrationAlgorithm == RIGID)
	{
		guiRigidReg->SetUp();
		guiRigidReg->Show();
	}
	/*else if (my_registrationAlgorithm == MUTUALINFO)
	{
		guiMI->SetUp(); 
		guiMI->Show();
	}*/
	
}
