//
// C++ Implementation: icp
//
// Description: Implementation of ICP (Iterative Closest Point) registration. ICP is a point set to point set registration, which has essentially the algorithm steps are: 1. Associate points by the nearest neighbor criteria. 2. Estimate the parameters using a mean square cost function. 3.Transform the points using the estimated parameters. 4. Iterate (re-associate the points and so on). ITK provides a point based registration framework that allows for easy use of ICP.
//
// Author:  <Marta Kersten>, (C) 2005
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "icp.h"

ICP::ICP()
{
	// set dynamic range of translations and rotations to
	setTranslationScale(1000.0); 
	setRotationScale(1.0); 

	// set registration parameters to defaults
	setNumberOfIterations(100); 
	setGradientTolerance(1e-4); 
	setValueTolerance(1e-4);
	setEpsilonFunction(1e-5); 
	setUseCostFunctionGradient(false);

	my_calculateDistanceMap = false; 

	// set up registration components
	my_metric = MetricType::New(); 
	my_transform = TransformType::New();
	my_optimizer = OptimizerType::New(); 
	my_registration = RegistrationType::New(); 

	my_fixedPointSet = PointSetType::New(); 
	my_movingPointSet = PointSetType::New();


}

ICP::ICP(double transScale, double rotScale, unsigned long numIter, double gradTolerance, double valueTolerance, double epsilon, bool useCostFuncGrad, bool distMap)
{
	// set dynamic range of translations and rotations
	setTranslationScale(transScale); 
	setRotationScale(rotScale); 
	
	// set up registration parameters
	setNumberOfIterations(numIter); 
	setGradientTolerance(gradTolerance); 
	setValueTolerance(valueTolerance);
	setEpsilonFunction(epsilon); 
	setUseCostFunctionGradient(useCostFuncGrad);

	setCalculateDistanceMap(distMap); 

	// set up registration components
	my_metric = MetricType::New(); 
	my_transform = TransformType::New();
	my_optimizer = OptimizerType::New(); 
	my_registration = RegistrationType::New(); 	
	std::cout<<"Number of fixed points is: " << my_fixedPointSet->GetNumberOfPoints(); 
	my_fixedPointSet = PointSetType::New(); 
	my_movingPointSet = PointSetType::New();

}


ICP::~ICP()
{
}


void ICP::Register()
{
	my_optimizer->SetUseCostFunctionGradient(my_useCostFunctionGradient);
	
	// Scale translation components of the Transform in the Optimizer
	OptimizerType::ScalesType scales(my_transform->GetNumberOfParameters()); 
		
	for(int i = 0; i < 3; i++) 
		scales[i] = 1.0/my_rotationScale; 

	for(int j = 3; j < 6; j++)
		scales[j] = 1.0/my_translationScale;
	
	my_optimizer->SetScales(scales); 
	my_optimizer->SetNumberOfIterations(my_numberOfIterations); 
	my_optimizer->SetValueTolerance(my_valueTolerance); 
	my_optimizer->SetGradientTolerance(my_gradientTolerance); 
	my_optimizer->SetEpsilonFunction(my_epsilonFunction); 

	// start from an identity transform (maybe change this as in  
	// normal case user should have better idea of transform
	my_transform->SetIdentity(); 

	my_registration->SetInitialTransformParameters(my_transform->GetParameters()); 

	// connect all the registration components
	my_registration->SetMetric(my_metric); 
	my_registration->SetOptimizer(my_optimizer); 
	my_registration->SetTransform(my_transform); 
	my_registration->SetFixedPointSet(my_fixedPointSet); 
	my_registration->SetMovingPointSet(my_movingPointSet); 


	//if calculating distance map do that now... 
	// IMPLEMENT LATER
	//if(distMap)
	//{
	//	CalculateDistanceMap();
	//}

	
	try
	{
		my_registration->StartRegistration(); 
	}
	catch (itk::ExceptionObject & e )
	{
		std::cout << e << std:: endl; 
	//	return EXIT_FAILURE; 	//change this at some point
	}

	// return transform->GetParameters(); 
	std::cout<< "Solution = " << my_registration->GetLastTransformParameters() << std::endl; 
}

void ICP::CalculateDistanceMap()
{

}

double ICP::epsilonFunction() const
{
  return my_epsilonFunction;
}


void ICP::setEpsilonFunction(double theValue)
{
  my_epsilonFunction = theValue;
}


double ICP::gradientTolerance() const
{
  return my_gradientTolerance;
}


void ICP::setGradientTolerance(double theValue)
{
  my_gradientTolerance = theValue;
}


unsigned long ICP::numberOfIterations() const
{
  return my_numberOfIterations;
}


void ICP::setNumberOfIterations(unsigned long theValue)
{
  my_numberOfIterations = theValue;
}


double ICP::rotationScale() const
{
  return my_rotationScale;
}


void ICP::setRotationScale(double theValue)
{
  my_rotationScale = theValue;
}


double ICP::translationScale() const
{
  return my_translationScale;
}


void ICP::setTranslationScale(double theValue)
{
  my_translationScale = theValue;
}


bool ICP::useCostFunctionGradient() const
{
  return my_useCostFunctionGradient;
}


void ICP::setUseCostFunctionGradient(bool theValue)
{
  my_useCostFunctionGradient = theValue;
}


double ICP::valueTolerance() const
{
  return my_valueTolerance;
}


void ICP::setValueTolerance(double theValue)
{
  my_valueTolerance = theValue;
}


bool ICP::calculateDistanceMap() const
{
  return my_calculateDistanceMap;
}


void ICP::setCalculateDistanceMap(bool theValue)
{
  my_calculateDistanceMap = theValue;
}
